<?php

namespace App\Lib;

use App\Constants\Status;
use App\Models\Invest;
use App\Models\Referral;
use App\Models\Transaction;
use App\Models\AdminNotification;

class TreeInvestLab
{
    /**
     * Instance of investor user
     *
     * @var object
     */
    private $user;

    /**
     * Plan which is purchasing
     *
     * @var object
     */
    private $project;

    /**
     * General setting
     *
     * @var object
     */
    private $setting;

    /**
     * Set some properties
     *
     * @param object $user
     * @param object $project
     * @return void
     */
    public function __construct($user, $project)
    {
        $this->user    = $user;
        $this->project = $project;
        $this->setting = gs();
    }

    /**
     * Invest process
     *
     * @param float $amount
     * @return void
     */
    public function invest($quantity)
    {
        $project     = $this->project;
        $user        = $this->user;
        $singlePrice = $project->investment / $project->total_quantity;
        $amount      = $quantity * $singlePrice;

        $user->balance       -= $amount;
        $user->total_invests += $amount;
        $user->save();

        $trx                       = getTrx();
        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $amount;
        $transaction->post_balance = $user->balance;
        $transaction->charge       = 0;
        $transaction->trx_type     = '-';
        $transaction->details      = 'Invested on ' . $project->name;
        $transaction->trx          = $trx;
        $transaction->remark       = 'invest';
        $transaction->save();

        $invest                 = new Invest();
        $invest->user_id        = $user->id;
        $invest->project_id     = $project->id;
        $invest->quantity       = $quantity;
        $invest->amount         = $amount;
        $invest->period         = ($project->lifetime == Status::LIFETIME) ? -1 : $project->repeat_time;

        $invest->time_name      = $project->timeSetting->name;
        $invest->hours          = $project->timeSetting->time;
        $invest->status         = Status::INVEST_RUNNING;
        $invest->capital_status = $project->capital_back;

        $invest->trx            = $trx;
        $invest->save();

        if ($this->setting->invest_commission == Status::YES) {
            $commissionType = 'invest_commission';
            self::levelCommission($user, $amount, $commissionType, $trx);
        }

        $interestInfo = $project->fixed_interest ? showAmount($project->fixed_interest) :  $project->minimum_interest . '% to ' . $project->maximum_interest . '%';

        notify($user, 'INVESTMENT', [
            'trx'          => $invest->trx,
            'amount'       => showAmount($amount),
            'quantity'     => $quantity,
            'project_name' => $project->name,
            'interest'     => $interestInfo,
            'harvest'      => convertHarvestTime($project->harvest_time),
            'time'         => $project->lifetime == 1 ? 'lifetime' : $project->repeat_time . ' times',
            'time_name'    => $project->timeSetting->name,
            'post_balance' => showAmount($user->balance),
        ]);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = showAmount($amount) . ' invested to ' . $project->name;
        $adminNotification->click_url = urlPath('admin.report.invest.details', $invest->id);
        $adminNotification->save();
    }

    /**
     * Give referral commission
     *
     * @param object $user
     * @param float $amount
     * @param string $commissionType
     * @param string $trx
     * @param object $setting
     * @return void
     */
    public static function levelCommission($user, $amount, $commissionType, $trx)
    {
        $meUser       = $user;
        $i            = 1;
        $level        = Referral::where('commission_type', $commissionType)->count();
        $transactions = [];
        while ($i <= $level) {
            $me    = $meUser;
            $refer = $me->referrer;
            if ($refer == "") {
                break;
            }

            $commission = Referral::where('commission_type', $commissionType)->where('level', $i)->first();
            if (!$commission) {
                break;
            }

            $com = ($amount * $commission->percent) / 100;
            $refer->balance += $com;
            $refer->save();

            $transactions[] = [
                'user_id'      => $refer->id,
                'amount'       => $com,
                'post_balance' => $refer->balance,
                'charge'       => 0,
                'trx_type'     => '+',
                'details'      => 'level ' . $i . ' Referral Commission From ' . $user->username,
                'trx'          => $trx,
                'remark'       => 'referral_commission',
                'created_at'   => now(),
            ];

            if ($commissionType == 'deposit_commission') {
                $comType = 'Deposit';
            } else {
                $comType = 'Invest';
            }

            notify($refer, 'REFERRAL_COMMISSION', [
                'amount'       => showAmount($com),
                'post_balance' => showAmount($refer->balance),
                'trx'          => $trx,
                'level'        => ordinal($i),
                'type'         => $comType,
            ]);

            $meUser = $refer;
            $i++;
        }

        if (!empty($transactions)) {
            Transaction::insert($transactions);
        }
    }

    /**
     * Capital return
     *
     * @param object $invest
     * @param object $user
     * @return void
     */

    public static function capitalReturn($invest)
    {
        $user = $invest->user;
        $user->balance += $invest->amount;
        $user->save();

        $invest->capital_back = Status::CAPITAL_BACK;
        $invest->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $invest->amount;
        $transaction->charge       = 0;
        $transaction->post_balance = $user->balance;
        $transaction->trx_type     = '+';
        $transaction->trx          = getTrx();
        $transaction->remark       = 'capital_return';
        $transaction->details      = showAmount($invest->amount) . ' capital back from ' . @$invest->project->name;
        $transaction->save();
    }
}
